module Equality where

  -- ----------------------------------------------------------------------
  -- * Definitions

  infix 5 _==_
  
  -- Equality
  data _==_ {A : Set} : A -> A -> Set where
    refl : {x : A} -> x == x

  -- ----------------------------------------------------------------------
  -- * Properties

  symm : ∀ {A : Set} -> {a b : A} -> a == b -> b == a
  symm refl = refl 
  
  trans : ∀ {A : Set} -> {a b c : A} -> a == b -> b == c -> a == c
  trans refl refl = refl
  
  context : ∀ {A : Set} {B : Set} -> {a b : A} -> (f : A -> B) -> a == b -> f a == f b
  context f refl = refl

  subst : ∀ {A : Set} {a b : A} (C : A -> Set) -> a == b -> C a -> C b
  subst C refl x = x

  cong : ∀ {A : Set} {B : Set} {C : Set} (f : A -> B -> C) -> {x y : A} -> {x' y' : B} -> x == y -> x' == y' -> f x x' == f y y'
  cong _ refl refl = refl

  -- ----------------------------------------------------------------------
  -- * Combinators for equational reasoning
  
  -- To prove a0 == an:
  -- 
  -- equational a0
  --         by reason1
  --     equals a1
  --         by reason2
  --     equals a2
  --        ...
  --         by reasonn
  --     equals an
  
  equational_ : ∀ {A : Set} -> (a : A) -> a == a
  equational_ a = refl
  infix 2 equational_
  
  by-equals : ∀ {A : Set} -> {a b : A} -> a == b -> (c : A) -> b == c -> a == c
  by-equals eq c reason = trans eq reason

  syntax by-equals eq c reason = eq by reason equals c
  infixl 1 by-equals

  -- 'symm' as a postfix operator.
  _reversed : ∀ {A : Set} -> {a b : A} -> a == b -> b == a
  _reversed = symm
  infixl 4 _reversed

  -- | Another name for 'refl' so that we can say "by definition"
  definition : ∀ {A : Set} -> {a : A} -> a == a
  definition = refl

  -- | Another name for 'refl', for use with tactics.
  auto : {A : Set} {x : A} -> x == x
  auto = refl

  {-# BUILTIN EQUALITY _==_ #-}
