/* semantics.c */
/* This is part of the source for the Core Join Calculus Compiler cjcc */
/* Copyright (C) 1996 Peter Selinger */
/* This is free software under the terms of the GNU General Public License. */

/* This file contains the definitions of the semantic values of
   non-terminals from the values of their components. For a
   description of the semantic types, see types.c */

char sacc[10000];	/* string accumulator */

processtype empty_process(void)
{
  processtype out_proc;

  out_proc.local = emptysem();
  out_proc.global = emptysem();
  out_proc.free = emptylist();
  return out_proc;
}

processtype proc_from_proc_message(in_proc, in_msg)
     processtype in_proc;
     processtype in_msg;
{
  processtype out_proc;
  
  out_proc.local = app(in_proc.local,in_msg.local);
  out_proc.global = app(in_proc.global,in_msg.global);
  out_proc.free = append(in_proc.free,in_msg.free); 
  return out_proc;
}

sem *receiver_to_code(receivertype *r)
{
  sem *code;
  list *rec;

  code=app(instr(r->thisdefa.d),instr("<"));
  rec=r->received;
  while (rec!=NULL) {
    code=app(code,rec->x);
    if ((rec=rec->next)!=NULL)
      code=app(code,instr(","));
  }
  code=app(code,instr(">"));
  return code;
}

processtype proc_from_name_def_proc(in_defname,in_def,in_proc)
     char *in_defname;
     definitiontype in_def; 
     processtype in_proc;                  /* def D in P */
{ 
  processtype out_proc;

  int n,m,i;
  list *defa, *free, *coll, *pattern, *var;
  list *def;
  sem *code;
  char **d, **f;
  int *a;

  defa = setify(in_def.defa);  /* list of defined var's of D + arities */ 
  n = length(defa);
  
  d=(char **)calloc(n,sizeof(char *));
  a=(int *)calloc(n,sizeof(int));
  
  if (d==NULL || a==NULL) {
    compile_time_err("Out of Memory");
    exit(1);
  }
  
  def = emptylist();       /* convert defined var's to two other formats: */
  for (i=0; i<n; i++) {
    d[i]=defa->x->dct->d;        /* array of names d[] = dv(D) */
    a[i]=defa->x->dct->a;  /* array of arities a[] */
    def=cons(instr(d[i]),def);    /* list of (names as semantic values) */
    defa=defa->next;
  }
  out_proc.free=minus(
		 append(listclone(in_proc.free),listclone(in_def.free))
		,def);		/* fv(def D in P) = (fv(P)+fv(D))-dv(D) */

  free = setify(minus(in_def.free,def));   /* compute fv(D)-dv(D) */
  m = length(free);
  
  f=(char **)calloc(m,sizeof(char *));
  
  if (f==NULL) {
    compile_time_err("Out of Memory");
    exit(1);
  }
  
  for (i=0; i<m; i++) {     /* and convert fv(D)-dv(D) to array f[] */
    f[i]=free->x->cct;
    free=free->next;
  }

  sprintf(sacc,
	  
	  "\n  /* make new incarnation of definition {%s}. */\n\n"
  	  "  { incarnation *inc;\n"
	  "    if ((inc=(incarnation *)calloc(1,sizeof(incarnation)))==NULL\n"
	  "         || (inc->q=(queue *)calloc(%d,sizeof(queue)))==NULL\n"
	  "         || (inc->v=(value *)calloc(%d,sizeof(value)))==NULL\n"
	  "         || (inc->arity=(int *)calloc(%d,sizeof(int)))==NULL)\n"
	  "      run_time_err(\"Out of Memory\");\n"
	  "    inc->code = def_%s;\n"
	  "    { int i;\n"
	  "      for (i=0; i<%d; i++)\n"
	  "        init_queue(&(inc->q[i]));\n"
	  "    }\n"
	  
	  ,in_defname,n,m,n,in_defname,n);
  code = instr(strclone(sacc));
  for (i=0; i<n; i++) {
    sprintf(sacc,
	    
	    "    inc->arity[%d]=%d;\n"
	    
	    ,i,a[i]); 
    code = app(code,instr(strclone(sacc)));
  }
  for (i=0; i<m; i++) {
    sprintf(sacc,
	    
	    "    inc->v[%d]=name_%s;\n"
	    
	    ,i,f[i]);
    code = app(code,instr(strclone(sacc)));
  }
  code = app(code,instr(
			
			"    { value "
			
			));  
  for (i=0; i<n; i++) {
    sprintf(sacc,
	    
	    "name_%s%c"
	    
	    ,d[i],(i==n-1 ? ';' : ','));
    code = app(code,instr(strclone(sacc)));
  }
  code = app(code,instr("\n"));
  for (i=0; i<n; i++) {
    sprintf(sacc,
	    
	    "      name_%s.type=CHANNEL;\n"
	    "      name_%s.c.chan.inc=inc;\n"
	    "      name_%s.c.chan.entry=%d;\n"
	    
	    ,d[i],d[i],d[i],i);
    code = app(code,instr(strclone(sacc)));
  }
  code = app(code,in_proc.local);
  code = app(code,instr(
			"    }\n"
			"  }\n"
			));
  out_proc.local=code;
  
  code=app(in_def.global,in_proc.global);
  sprintf(sacc,

	  "/* definition {%s} */\n\n"
	  "void def_%s(incarnation *inc, int entry) {\n"
	  
	  ,in_defname,in_defname);
  code=app(code,instr(strclone(sacc)));
  
  coll=in_def.collector;   /* coll points to the next (basicdef as a
			      semantic value) */
  while(coll!=NULL) {
    int flag=0;                      /* each J --> P' */

    code=app(code,instr("\n  /* "));
    pattern=coll->x->bct->pattern;
    while (pattern!=NULL) {
      code=app(code,receiver_to_code(pattern->x->rct));
      if ((pattern=pattern->next)!=NULL)
	code=app(code,instr("|"));
    }
					
    code=app(code,instr(
			" --> ... */\n\n"
			"  if ("
			));
    for (i=0;i<n;i++) {
      /* how often does d[i] appear in this J? */
      int count=0;
      pattern=coll->x->bct->pattern;
      while (pattern!=NULL) {
	if (streq(d[i],pattern->x->rct->thisdefa.d))
	  count++;
	pattern=pattern->next;
      }
      if (count) {
	if (flag) {
	  code=app(code,instr(
			      " &&\n      "		      
			      ));
	}
	sprintf(sacc, 
		
		"queue_length(inc->q+%d) >= %d"
		
		,i,count*(a[i] ? a[i] : 1));
	code=app(code,instr(strclone(sacc)));
	flag=1;
      }
    }
    code=app(code,instr(                 /* declare all names in P': */
			") {\n"
			"     value "
			));
    var=coll->x->bct->received;    /* received names rv(J) */
    if (var!=NULL) {
      while (var!=NULL) {
	sprintf(sacc,
		
		"name_%s,"
		,var->x->cct);
	code=app(code,instr(strclone(sacc)));
	var=var->next;
      }
      code=app(code,instr(
			  "\n           "
			  ));
    }
    var=coll->x->bct->free;

    var=setify(minus(var,def)); /* free names fv(J-->P') that are not dv(D) */ 
    
    if (var!=NULL) {
      while (var!=NULL) {
	sprintf(sacc,
		
		"name_%s,"
		,var->x->cct);
	code=app(code,instr(strclone(sacc)));
	var=var->next;
      }
      code=app(code,instr(
			  "\n           "
			  ));
    }
    for (i=0; i<n; i++) {           /* those defined var's dv(D) 
				       that are not received in rv(J) */
      if (!member(instr(d[i]),coll->x->bct->received)) {
	sprintf(sacc,
	      
		"name_%s%c"
		,d[i],(i==n-1 ? ';' : ','));
	code=app(code,instr(strclone(sacc)));
      }
    }
    code=app(code,instr(
			"\n"
			));          /* done declaring variables */
    pattern=coll->x->bct->pattern;   /* now instantiate them */
    while (pattern!=NULL) {  /* first, for each d<v..v> in J, pop v..v
				from stack d */
      int j;
      var=pattern->x->rct->received; 
      for (j=0; 
	   !streq(d[j],pattern->x->rct->thisdefa.d);
	   j++) ;
      if (var==NULL) {
	/* 0-ary channel needs popping dummy signal */
	sprintf(sacc,
		
		"     pop(&dummy,inc->q+%d); /* 0-ary channel */\n"
		
		,j);
	code=app(code,instr(strclone(sacc)));
      } else {
	while (var!=NULL) {
	  sprintf(sacc,
		  
		  "     pop(&name_%s,inc->q+%d);\n"
		  
		  ,var->x->cct,j);
	  code=app(code,instr(strclone(sacc)));
	  var=var->next;
	}
      }
      pattern=pattern->next;
    }
    for (i=0; i<n; i++) {  /* instantiate dv(D) according to
			      current incarnation, except those in rv(J) */
      if (!member(instr(d[i]),coll->x->bct->received)) {
	sprintf(sacc,
		
		"     name_%s.c.chan.entry=%d;\n"
		"     name_%s.c.chan.inc=inc;\n"
		"     name_%s.type=CHANNEL;\n"
		
		,d[i],i,d[i],d[i]);
	code=app(code,instr(strclone(sacc))); 
      }
    }
    var=coll->x->bct->free;           /* instantiate free var's fv(D)-rv(J)
					 according to current incarnation */
    var=setify(minus(var,def));
    while (var!=NULL) {
      int j;
      for (j=0;
	   j<m && !streq(f[j],var->x->cct);
	   j++);
      sprintf(sacc,
	      "     name_%s=inc->v[%d];\n"
	      ,var->x->cct,j);
      code=app(code,instr(strclone(sacc)));
      var=var->next;
    }
    code=app(code,coll->x->bct->proccode);
    
    code=app(code,instr(
			"  }\n"
			));
    coll=coll->next;
  }
  code=app(code,instr(
		      "}\n\n"
		      ));
  out_proc.global=code;
  return out_proc;
}

int defname_counter=0;

processtype proc_from_def_proc(in_def, in_proc)
definitiontype in_def; 
processtype in_proc;
{
  sprintf(sacc, "_%d", defname_counter++);
  return proc_from_name_def_proc(strclone(sacc),in_def,in_proc);
}

processtype message_from_name_explist(char *in_name, explisttype in_exl)
{ 
  processtype out_message;         /* x<v..v> */
  int i;
  sem *code;

  code=app(instr("\n  /* send message "),instr(in_name));
  code=app(code,instr("<"));
  code=app(code,in_exl.term);
  sprintf(sacc, 
	  
	  "> */\n\n"
	  "  if (name_%s.type!=WILDCARD) {\n"
	  "    if (name_%s.type!=CHANNEL)\n"
	  "      run_time_err(\"Can't send message along %s: not a channel\");\n"
	  "    { incarnation *inc = name_%s.c.chan.inc;\n"
	  "      int entry = name_%s.c.chan.entry;\n"
	  "      queue *xq = &(inc->q[entry]);\n"
	  "      if (inc->arity[entry] != %d)\n"
	  "        run_time_err2(\"Attempt to send %d-ary message %s<"

	  ,in_name,in_name,in_name,in_name,in_name,in_exl.n,in_exl.n,in_name);
  code=app(code,instr(strclone(sacc)));
  code=app(code,in_exl.term);
  code=app(code,instr("> along \%d-ary channel\",inc->arity[entry]);\n"));
  code=app(code,in_exl.code);

  if (in_exl.n==0) {
    /* 0-ary channel needs dummy pushed */
    code=app(code,instr(
	  "      push(dummy,xq); /* 0-ary channel */\n"
	  "#ifdef DEBUG\n"
	  "      fprintf(stderr, \"Pushing dummy onto stack \%d\\n\",entry);\n"
          "#endif\n"
    ));
  }
  sprintf(sacc,

	  "#ifdef DEBUG\n"
	  "      fprintf(stderr, \"Sending message along channel %s \");\n"
          "      printvalue(stderr, name_%s);\n"
          "      fprintf(stderr, \"\\n\");\n"
	  "#endif\n"
	  "      inc->code(inc,entry);\n"
	  "    }\n"
	  "  }\n"
	  
	  ,in_name);
  code=app(code,instr(strclone(sacc)));
  out_message.local=code;
  out_message.global=emptysem();
  out_message.free=cons(instr(strclone(in_name)),in_exl.free);
  return out_message;
}

explisttype empty_explist(void) 
{
  explisttype out_nl;

  out_nl.code=emptysem(); 
  out_nl.n=0; 
  out_nl.free=emptylist(); 
  out_nl.term=emptysem();
  return out_nl;
}

explisttype explist_from_exp_explist(exptype in_exp, explisttype in_nl)
{
  explisttype out_nl;         /* v...v */
  
  out_nl.code=app(in_exp.precode,instr("      push("));
  out_nl.code=app(out_nl.code,in_exp.funcode);
  out_nl.code=app(out_nl.code,instr(",xq);\n"
	  "#ifdef DEBUG\n"
	  "      fprintf(stderr, \"Pushing argument \");\n"
	  "      printvalue(stderr, "
				    ));
  out_nl.code=app(out_nl.code,in_exp.funcode);
  out_nl.code=app(out_nl.code,instr(
	  ");\n"
	  "      fprintf(stderr, \" onto stack \%d\\n\",entry);\n"
	  "#endif\n"
				    ));
  out_nl.code=app(out_nl.code,in_nl.code);
  out_nl.n=in_nl.n+1; 
  out_nl.free=append(in_exp.free,in_nl.free);
  out_nl.term=in_nl.term;
  if (in_nl.n != 0)
    out_nl.term=app(instr(","),out_nl.term);
  out_nl.term=app(in_exp.term,out_nl.term);
  return out_nl;
}

exptype exp_from_name(char *in_name)
{
  exptype out_exp;

  out_exp.precode = instr("");
  sprintf(sacc, "name_%s",in_name);
  out_exp.funcode = instr(strclone(sacc));    
  out_exp.free = singleton(instr(strclone(in_name)));
  out_exp.term = instr(in_name);
  return out_exp;
}

exptype exp_from_int(int in_int)
{
  exptype out_exp;
  
  sprintf(sacc,  
	  "      tmp_val.type=INT;\n"
	  "      tmp_val.c.i=%d;\n"
	  ,in_int);
  out_exp.precode = instr(strclone(sacc));
  out_exp.funcode = instr("tmp_val");
  out_exp.free = emptylist();
  sprintf(sacc,"%d",in_int);
  out_exp.term = instr(strclone(sacc));
  return out_exp;
}

exptype exp_from_str(char *in_str)
{
  exptype out_exp;
  
  sprintf(sacc,  
	  "      tmp_val.type=STR;\n"
	  "      tmp_val.c.s=\"%s\";\n"
	  ,in_str);
  out_exp.precode = instr(strclone(sacc));
  out_exp.funcode = instr("tmp_val");
  out_exp.free = emptylist();
  out_exp.term = app(app(instr("\\\""),instr(in_str)),instr("\\\""));
  return out_exp;
}

exptype exp_from_boole(int in_b)
{
  exptype out_exp;
  
  sprintf(sacc,  
	  "      tmp_val.type=BOOLE;\n"
	  "      tmp_val.c.b=%d;\n"
	  ,in_b);
  out_exp.precode = instr(strclone(sacc));
  out_exp.funcode = instr("tmp_val");
  out_exp.free = emptylist();
  if (in_b)
    out_exp.term = instr("true");
  else
    out_exp.term = instr("false");
  return out_exp;
}

exptype exp_from_wildcard(void)
{
  exptype out_exp;
  
  out_exp.precode = instr("      tmp_val.type=WILDCARD;\n");
  out_exp.funcode = instr("tmp_val");
  out_exp.free = emptylist();
  out_exp.term = instr("*");
  return out_exp;
}

definitiontype definition_from_basicdef(in_bd)
     basicdeftype in_bd;            /* D = J...J-->P */
{    
  definitiontype out_def;

  out_def.defa=listclone(in_bd.defa);
  out_def.free=listclone(in_bd.free);
  out_def.collector=singleton(inbasicdef(in_bd));
  out_def.global=in_bd.global;
  return out_def;
}

definitiontype definition_from_definition_basicdef(in_def, in_bd)
     definitiontype in_def;
     basicdeftype in_bd;            /* D = D and J...J-->P */
{
  definitiontype out_def;

  { char *a;
    int n, m;       /* check that d's occur with consistent arities */

    if (!consistent(in_def.defa,in_bd.defa,&a,&n,&m)) {
      sprintf(sacc, "Defined variable %s is defined with "
	      "arities %d and %d in a join pattern",a,n,m);
      compile_time_err(sacc);
      return;
    }
  }

  out_def.defa=append(in_def.defa,listclone(in_bd.defa));
  out_def.free=append(in_def.free,listclone(in_bd.free));
  out_def.collector=cons(inbasicdef(in_bd),in_def.collector);
  out_def.global=app(in_def.global,in_bd.global);

  return out_def;
}

basicdeftype basicdef_from_jp_process(in_jp,in_proc)
     joinpatterntype in_jp;
     processtype in_proc;               /* J..J --> P */
{
  basicdeftype out_bd;
  
  out_bd.proccode=in_proc.local;
  out_bd.free=minus(in_proc.free,in_jp.received);
  out_bd.received=in_jp.received;
  out_bd.pattern=in_jp.pattern;
  out_bd.global=in_proc.global;
  out_bd.defa=in_jp.defa;
  
  return out_bd;
}

joinpatterntype joinpattern_from_receiver(in_rc)
     receivertype in_rc;            /* J = x<v..v> */
{
  joinpatterntype out_jp;

  out_jp.received=listclone(in_rc.received);
  out_jp.defa=singleton(indefa(in_rc.thisdefa));
  out_jp.pattern=singleton(inreceiver(in_rc));
  
  return out_jp;
}

joinpatterntype jp_from_jp_receiver(in_jp,in_rc)
     joinpatterntype in_jp;
     receivertype in_rc;            /* J = J | x<v..v> */
{
  joinpatterntype out_jp;

  if (!disjoint(in_jp.received,in_rc.received)) {
    compile_time_err("Parameters in "
		     "join patterns are not disjoint");
    return;
  }
  out_jp.received=append(in_jp.received,listclone(in_rc.received));
  { char *a;
    int n,m;
    if (!consis(&in_rc.thisdefa,in_jp.defa,&a,&n,&m)) {
      sprintf(sacc, "Defined variable %s is defined with "
	      "arities %d and %d in a join pattern",a,n,m);
      compile_time_err(sacc);
      return;
    }
  }	
  out_jp.defa=cons(indefa(in_rc.thisdefa),in_jp.defa);
  out_jp.pattern=cons(inreceiver(in_rc),in_jp.pattern);
  
  return out_jp;
}

receivertype receiver_from_name_parameterlist(in_name,in_pl) 
     char *in_name;
     parameterlisttype in_pl;
{
  receivertype out_jp;

  out_jp.thisdefa.d=in_name;
  out_jp.thisdefa.a=in_pl.n;
  out_jp.received=in_pl.received;
  
  return out_jp;
}

parameterlisttype empty_parameterlist(void) {
  parameterlisttype out_pl;
  
  out_pl.received=emptylist(); 
  out_pl.n=0;
  return out_pl;
}

parameterlisttype pl_from_name_pl(char *in_name, parameterlisttype in_pl)
{
  parameterlisttype out_pl;

  sem *s;
  s=instr(in_name);
  if (member(s,in_pl.received)) {
    compile_time_err("Parameters in "
		     "join patterns are not disjoint");
    return;
  }
  out_pl.received=cons(s,in_pl.received); 
  out_pl.n=in_pl.n+1;
  
  return out_pl;
}

sem *program_from_process(processtype in_proc)
{ list *l;
  sem *code;
  list *g;
  list *gn;
  
  gn=doubleton(instr("write"),instr("readint"));
  gn=append(gn,doubleton(instr("equal"),instr("plus")));
  gn=append(gn,doubleton(instr("minus"),instr("times")));
  gn=append(gn,doubleton(instr("div"),instr("less")));
  gn=append(gn,singleton(instr("if")));
  
  if ((g=minus(in_proc.free,gn)) != NULL) {
    longlistprint(stderr,g);
    fprintf(stderr,"\n");
    compile_time_err("There are undefined global names");
    return;
  }

  code=instr(
#include "header.c"
		);
  code=app(code,instr(
"void sendalong0(value name_channel) {\n"
		      ));
  code=app(code,message_from_name_explist("channel",
					  empty_explist()).local);
  code=app(code,instr(
"}\n\n"
"void sendalong1(value name_channel,value name_sendee) {\n"
		      ));
  code=app(code,message_from_name_explist("channel",
	  explist_from_exp_explist(exp_from_name("sendee"),
				   empty_explist()
				   )).local);
  code=app(code,instr(
"}\n\n"
#include "serverdef.c"
		      ));
  code=app(code,in_proc.global);
  code=app(code,instr(                   "main()\n{\n"
#include "serverinc.c"
		      ));
  code=app(code,in_proc.local);
  code=app(code,instr(
		                         "  }\n}\n\n"
		      ));
  return code;
} 
