/* main.c */
/* This is part of the source for the Core Join Calculus Compiler cjcc */
/* Copyright (C) 1996 Peter Selinger */
/* This is free software under the terms of the GNU General Public License. */

#define VERSION "0.5"

#include <stdio.h>
#include <string.h>

int verbose=0;
char *outfile=NULL;
int global_error=0;
char *global_input_file;

char *default_outfile(char *infile)
{
  static char outfile[42];
  char *p;

  if (infile[0]=='-')
    return "-";

  /* if infile has suffix .j*, replace suffix by .c, else append .c */

  strncpy(outfile,infile,40);
  if (NULL==(p=strrchr(outfile,'.')) || p[1]!='j')
    return strcat(outfile,".c");
  else {
    p[1]='c';
    p[2]='\0';
    return outfile;
  }
}

char *strip_arg0(char *arg0)
{
  char *p;

  /* strip path name from arg0: e.g. /bin/cjcc into cjcc */ 

  if (NULL==(p=strrchr(arg0,'/')) || p[1]=='\0')
    return arg0;
  else 
    return p+1;
}

int license() {
  fprintf(stderr,
	  "Core Join Calculus Compiler (cjcc), Version %s.\n"
	  "Copyright (C) 1996 Peter Selinger.\n"
	  "This is free software under the terms of the GNU "
	  "General Public License.\n\n", VERSION);
}

int main(int argc, char *argv[])
{
  char *p;
  char *infile=NULL;
  char *arg0;
  FILE *fout;
  int using_stdin, using_stdout;

  arg0=strip_arg0(argv[0]);      /* remember program name */
  ++argv, --argc;         
    
  while (argc > 0) {             /* read arguments */
    p = *argv;
    ++argv, --argc;
    if (p[0] == '-' && p[1] != '\0') {           /* '-x' must be an option */
      p++;                       /* strip off the - */
      switch(p[0]) {
      case 'o':                  /* -o output file */
	if (argc<=0) {
	  fprintf(stderr, "No output file specified after -o\n");
	  usage(arg0);
	}
	outfile=*argv;
	++argv, --argc;
	break;
      case 'v':                  /* -v verbose */
	verbose++;
	if (1==verbose) license();
	break;
      case 'h':                  /* -h help */
	verbose++;
	if (1==verbose) license();
	help(arg0);
	usage(arg0);
	break;
      default: /* invalid option */
	fprintf(stderr,"Illegal option: -%s\n", p);
	usage(arg0);
	break;
      }
    } else { /* not an option: must be an input file */
      infile=p;
    }
  }
  if (infile==NULL) {
    fprintf(stderr, "No input file specified\n");
    usage(arg0);
  }

  /* done with reading options; syntax seemed okay */

  if (outfile==NULL) {  /* construct default output file name */
    outfile=default_outfile(infile);
  }
  using_stdin=using_stdout=0;
  if (infile[0]=='-') {  /* determine whether stdin/stdout used */
    infile="stdin";
    using_stdin=1;
  }
  if (outfile[0]=='-') {
    outfile="stdout";
    using_stdout=1;
  }
  
  if (verbose) fprintf(stderr, "Input file: %s, output file: %s\n",
		       infile, outfile);

  if (using_stdin)    /* open stream yyin */
    yyin=stdin;
  else if (NULL==(yyin=fopen(infile, "r"))) {
    fprintf(stderr,"Cannot open %s for input.\n",infile);
    exit(1);
  }

  global_input_line=1;       /* so that error messages know current line */
  global_input_file=infile;  /* so that error messages know current file */

  if (verbose) fprintf(stderr,"Reading input from %s ... ",infile);
  if (verbose) fprintf(stderr,"Parsing ... \n");

  yyparse();         /*********** PARSE ************/

  if (!using_stdin) {
    if (verbose) fprintf(stderr,"Closing %s\n",infile);
    fclose(yyin);
  }

  if (global_error) {   /* Do not write output if parse error occurred */
    if (verbose) fprintf(stderr,"There were some parse errors.\n");
    exit(1); 
  }
  if (using_stdout)   /* open stream fout */
    fout=stdout;
  else if (NULL==(fout=fopen(outfile,"w"))) {
    fprintf(stderr,"Cannot open %s for output.\n",outfile);
    exit(1);
  } 

  if (verbose) fprintf(stderr,"Writing output to %s ...\n", outfile);

  semprint(fout,global_code);

  if (!using_stdout) {
    if (verbose) fprintf(stderr,"Closing %s\n",outfile);
    fclose(fout);
  }
}

yyerror (char *s)  /* Called by yyparse on error */
{
  fprintf(stderr,"%s:%d:  %s\nwhile parsing ... ", 
	  global_input_file, global_input_line, s);
  print_last_N_tokens();
  fprintf(stderr,"\n\n");
  global_error=1;
}

compile_time_err(char *s) {
  fprintf(stderr, "Semantic error: \n");
  yyerror(s);
  exit(1);
}

#define NN 15
char nn_memory[NN][10];
int nn_next=1;

remember(char *str) {  /* keeps track of the last NN tokens of the input 
			 stream for use in error messages */
  if (nn_next>=NN) nn_next=0;
  strncpy(nn_memory[nn_next],str,9);
  nn_next++;
}

print_last_N_tokens(void) {
  int i=nn_next+1;
  
  do {
    if (i>=NN) i=0;
    if (i+1==nn_next) fprintf(stderr," ***here***  "); 
    if (nn_memory[i][0]!='\0') fprintf(stderr,"%s ",nn_memory[i]);
    i++;
  } while (i!=nn_next);
}

int usage(char *p)
{
  fprintf(stderr,"Usage: %s [-v[erbose]] [-o outfile] infile\n"
	  "       %s -h[elp]\n"
	  "       infile is - for stdin, outfile is - for stdout.\n", p, p);
  exit(0);
}

int help(char *s)
{
  fprintf(stderr,
"The Core Join Calculus is an experimental programming language described in\n"
"\n"
"[1] C. Fournet, G. Gonthier.  \n"
"    The reflexive CHAM and the join-calculus.  To appear in POPL'96.  \n"
"    http://pauillac.inria.fr/~fournet/papers/popl-96.ps.gz\n"
"\n"
"[2] C. Fournet, G. Gonthier, J-J. Levy, L. Maranget, D. Remy. \n"
"    A Calculus of Mobile Agents.  To appear in CONCUR'96.\n"
"    http://pauillac.inria.fr/~fournet/papers/concur-96.ps.gz\n"
"\n"
"'cjcc' is an experimental compiler of the core join calculus into C.\n"
"To compile a join calculus file, first use 'cjcc' to generate C code,\n"
"then use a C-compiler to get an executable file. For example, to compile\n"
"and run the file 'numbers.j', type\n"
"\n"
"        %s -o numbers.c numbers.j\n"
"        gcc -o numbers numbers.c\n"
"        numbers\n"
"\n"
"Here is a brief summary of the syntax used by cjcc:\n"
"\n"
"Values v ::= x | 123 | \"abc\" | true | false | _\n"
"\n"
"Processes P ::= x<v1,...,vn>\n"
"                {}\n"
"                {P}\n"
"                P | P\n"
"                def D in P\n"
"                def /* name */ D in P\n"
"\n"
"Definitions D ::= J --> P\n"
"                  D and D\n"
"\n"
"Join Patterns J ::= x<y1,...,yn>\n"
"                    J | J         (all yi distinct names)\n"
"\n"
"System calls: write <value,continuation,exception>\n"
"              readint <int_result,exc>\n"
"              equal <int1,int2,boole_result,exc> similar less\n"
"              plus <int1,int2,int_result,exc>    similar minus,times,div\n"
"              if <boole,then_cont,else_cont,exc>\n\n",s);
}

